/* $Id: intr2.c,v 1.12 1998/10/16 22:55:59 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* This file contains code showing how to set up SICL and an E143x for
   interrupt-driven data collection.

   The measurement sets up the E143x module to interrupt on any of
   three different things:

   1) Data ready for host to read
   2) Measurement Error - this means a FIFO overflow occurred
   3) Overload change - one or more input channels changed from
      overloaded to not overloaded, or vice versa.

   In response to any of these interrupts, a message is printed.  In
   addition, for a data ready interrupt, the program reads out a scan
   of data.  For a overload change interrupt, the program checks and
   prints out the overload status of all input channels.

   This program can be used to verify that the module continues to
   send data to the host even after a FIFO overflow is detected.  It
   is not until the host reads all of the data out of the FIFO that
   the e1432_block_available function returns ERR1432_FIFO_OVERRUN and
   stops indicating that data is ready.  To do this verification, do
   the following:

   * Run intr2.
   * To force a FIFO overflow, hit "control-S" while the program is
     running, which should stop the program from printing messages.
     This will eventually cause the program to block, and stop reading
     data.
   * Wait awhile, long enough that a FIFO overflow will happen.  You
     won't get any message, because of the "control-S", so just wait.
     For a module with 8 channels and 32 MB of DRAM this takes at
     least 41 seconds.
   * Hit "control-Q" to re-enable output from the program.
   * You should see a "FIFO overflow detected" message almost
     immediately.
   * You should see the program continue to read out data, even though
     the FIFO has overflowed, until the FIFO is empty, and then a FIFO
     overflow message from e1432_block_available.
*/

#include <stdio.h>		/* For printf */
#include <stdlib.h>		/* For strtol */
#include <string.h>		/* For strrchr */
#include <unistd.h>		/* For sleep */
#include <sicl.h>		/* For SICL functions */
#include "e1432.h"
#include "err1432.h"

/* Most of this program currently supports only one module */
#define	NMOD_MAX	1
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	8192	/* Max FFT blocksize is 8192 */

#define	DEBUG(s)	s

/* Wrap this around all the many function calls which might fail.  Use
   "!=" because sicl has positive error numbers. */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: intr2.c,v 1.12 1998/10/16 22:55:59 ericb Exp $";
static const char *progname;

/* Make these global so the interrupt handler can get to them easily */
static struct global_struct_type
{
    E1432ID hw;
    SHORTSIZ16 chan_list[NCHAN_MAX];
    SHORTSIZ16 laddr;
    int     group, nchan, fifo_overflow, data_done, freq, no_time;
    INST    irq_id;
} gs;

static int
handle_meas_error(struct global_struct_type *g, long sec)
{
    if ((sec & E1432_IRQ_MEAS_ERROR) != 0)
    {
	DEBUG((void) printf("*** Fifo overflow detected ***\n"));
	g->fifo_overflow = 1;
    }
    return 0;
}

static int
handle_data(struct global_struct_type *g, long sec)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    SHORTSIZ16 status;
    int     i, scan_count;

    /* See if there is data.  Read up to four scans of data */
    scan_count = 0;
    while (scan_count++ < 4)
    {
	status = e1432_block_available(g->hw, g->group);

	/* Check for mismatch between e1432_block_available and sec */
	if (scan_count == 1)
	{
	    if (status <= 0 && (sec & E1432_IRQ_BLOCK_READY) != 0)
		(void) printf("Error: block ready irq_handler sec "
			      "and e1432_block_available mismatch\n");
	    if (status == ERR1432_FIFO_OVERRUN && !g->fifo_overflow)
		(void) printf("Error: fifo overflow without "
			      "interrupt first\n");
	}

	/* Print what happenned */
	if (status < 0 && status != ERR1432_FIFO_OVERRUN)
	    (void) printf("Error: e1432_block_available error %d\n",
			  status);
	else if (status > 0)
	    DEBUG((void) printf("Block available found!\n"));

	/* Tell foreground routine we're done */
	if (status == ERR1432_FIFO_OVERRUN)
	    g->data_done = 1;

	/* If error or no more data, return */
	if (status <= 0)
	    return status;

	/* Read one scan of data */
	if (!(g->freq && g->no_time))
	    for (i = 0; i < g->nchan; i++)
	    {
		CHECK(e1432_read_float64_data(g->hw, g->chan_list[i],
					      E1432_TIME_DATA, buffer,
					      BLOCKSIZE, NULL, &count));
		if (count != BLOCKSIZE)
		{
		    (void) fprintf(stderr, "Actual count was %d\n", count);
		    return -1;
		}
	    }
	if (g->freq)
	    for (i = 0; i < g->nchan; i++)
	    {
		CHECK(e1432_read_float64_data(g->hw, g->chan_list[i],
					      E1432_FREQ_DATA, buffer,
					      BLOCKSIZE, NULL, &count));
		if (count != BLOCKSIZE)
		{
		    (void) fprintf(stderr, "Actual count was %d\n", count);
		    return -1;
		}
	    }
    }

    status = e1432_block_available(g->hw, g->group);
    /* Tell foreground routine we're done */
    if (status == ERR1432_FIFO_OVERRUN)
	g->data_done = 1;

    return 0;
}

static int
handle_ovld(struct global_struct_type *g, long sec)
{
    SHORTSIZ16 common[NCHAN_MAX];
    SHORTSIZ16 diff[NCHAN_MAX];
    SHORTSIZ16 half[NCHAN_MAX];
    int     i;

    if ((sec & E1432_IRQ_OVERLOAD_CHANGE) != 0)
    {
	(void) printf("Overload change detected\n");
	CHECK(e1432_check_overloads(g->hw, g->group, NULL,
				    common, diff, half));

	for (i = 0; i < g->nchan; i++)
	    (void) printf("Chan %2d:  comm %d  diff %d  half %d\n", i,
			  common[i], diff[i], half[i]);
    }

    return 0;
}

static int
handle_reenable(struct global_struct_type *g)
{
    CHECK(e1432_reenable_interrupt(g->hw, g->group));
    return 0;
}

static void
irq_handler(INST id, long reason, long sec)
{
    DEBUG((void) printf("irq_handler called\n"));

    if (id != gs.irq_id)
	(void) printf("Error: irq_handler got wrong id\n");
    if (reason != I_INTR_VXI_SIGNAL)
	(void) printf("Error: irq_handler got wrong reason\n");
    if ((sec & E1432_IRQ_STATUS_LADDR_MASK) != gs.laddr)
	(void) printf("Error: irq_handler got wrong laddr\n");

    (void) handle_meas_error(&gs, sec);
    (void) handle_ovld(&gs, sec);
    (void) handle_data(&gs, sec);
    (void) handle_reenable(&gs);
}

static int
irq_setup(int laddr, int *line, INST *irq_id)
{
    struct vxiinfo info;
    unsigned long slot0_laddr;
    char    addr[16];
    INST    id;
    int     i;

    /* Get the interrupt line to use */
    id = iopen("vxi");
    if (id == 0)
    {
	(void) fprintf(stderr, "Error: iopen returned %d\n", id);
	return -1;
    }
    CHECK(ivxibusstatus(id, I_VXI_BUS_LADDR, &slot0_laddr));
    CHECK(ivxirminfo(id, slot0_laddr, &info));
    CHECK(iclose(id));
    for (i = 0; i < 8; i++)
	if (info.int_handler[i] != -1)
	{
	    *line = info.int_handler[i];
	    break;
	}
    if (i == 8)
    {
	(void) fprintf(stderr, "Error: no interrupt lines available\n");
	return -1;
    }
    DEBUG((void) printf("Using VME interrupt line %d\n", *line));

    /* Set up the interrupt handler routine */
    (void) sprintf(addr, "vxi,%d", laddr);
    *irq_id = iopen(addr);
    if (*irq_id == 0)
    {
	(void) fprintf(stderr, "Error: iopen returned %d\n", *irq_id);
	return -1;
    }
    CHECK(ionintr(*irq_id, irq_handler));
    CHECK(isetintr(*irq_id, I_INTR_VXI_SIGNAL, 1));

    return 0;
}

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group, int laddr, int freq, int no_time, INST *irq_id)
{
    int     line;

    /* Set up parameters */
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_data_mode(hw, group, E1432_CONTINUOUS_MODE));
    CHECK(e1432_set_range(hw, group, 1));
    if (freq)
    {
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));
	if (no_time)
	    CHECK(e1432_set_enable(hw, group, E1432_ENABLE_TYPE_TIME,
				   E1432_ENABLE_OFF));
    }

    /* Set up interrupts */
    CHECK(irq_setup(laddr, &line, irq_id));
    CHECK(e1432_set_interrupt_priority(hw, group, line));
    CHECK(e1432_set_interrupt_mask(hw, group,
				   E1432_IRQ_BLOCK_READY |
				   E1432_IRQ_OVERLOAD_CHANGE |
				   E1432_IRQ_MEAS_ERROR));

    return 0;
}

static int
run(E1432ID hw, int group, int *fifo_overflow, int *data_done)
{
    /* Start measurement */
    *fifo_overflow = 0;
    *data_done = 0;
    CHECK(e1432_init_measure(hw, group));

    /* Run until fifo overflow */
    while(!*data_done)
	(void) sleep(10000);
    /* As a test, allow for subsequent interrupts to get noticed */
    (void) sleep(1);

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-ftuV] [-L laddr] [-n nchan]\n"
		   "\t-f: Set up for FFT data\n"
		   "\t-L: Logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-t: When doing FFT data, disable time data\n"
		   "\t-u: Print this usage message\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    char   *p;
    int     c;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    gs.freq = 0;
    gs.laddr = 8;
    gs.nchan = -1;		/* Meaning use all channels */
    gs.no_time = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "fL:n:tuV")) != -1)
	switch (c)
	{
	case 'f':
	    gs.freq = 1;
	    break;
	case 'L':
	    gs.laddr = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || gs.laddr < 0 || gs.laddr > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    gs.nchan = strtol(optarg, &p, 0);
	    if (optarg == p || gs.nchan < -1 || gs.nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 't':
	    gs.no_time = 1;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Run the measurement */
    if (init(1, &gs.laddr, &gs.hw, &gs.group, &gs.nchan, gs.chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(gs.hw, gs.group, gs.laddr, gs.freq, gs.no_time, &gs.irq_id) < 0)
	return EXIT_FAILURE;
    if (run(gs.hw, gs.group, &gs.fifo_overflow, &gs.data_done) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
